//
// Copyright (C) ValoFly GmbH - All Rights Reserved
//

#include <TetherCom/TetherCom.h>

#include <string>
#include <vector>
#include <iostream>

void printHeader()
{
	std::cout << "###########################################" << std::endl;
	std::cout << "# ValoFly TetherCom Library usage example #" << std::endl;
	std::cout << "###########################################" << std::endl << std::endl;
	std::cout << "This example applications shows, how ValoFly TetherCom library can be used." << std::endl << std::endl << std::endl;
}

/**
* Print help
*/
void printHelp()
{
	printHeader();
}

/**
* Parse given arguments of program call
*
* Parse given arguments and call help function if help flag is part of given arguments
*
* @throw "Runtime Error" Throws runtime error on unknown flags or if help function was called
*/
void parseArguments(int argc, char* argv[])
{
	int configArgPos = -1;
	std::string configFile;

	for (int i = 0; i < argc; i++)
	{
		if (std::string(argv[i]).compare("-h") == 0 || std::string(argv[i]).compare("/h") == 0)
		{
			printHelp();
			std::exit(EXIT_SUCCESS);
		}
	}
}

/**
* Example of TetherCom library usage
*
* This is an example application showing usage of the TetherCom library.
*
* It shows:
*   - create an instance of library object
*   - get list of serial devices and print them
*   - open a connection to ValoFly Tether.Solutions ground station
*   - request current system status information and print them
*   - setup a configuration parameter set and transmit it to ValoFly Tether.Solutions ground station
*   - set single configuration parameters
*   - activate and deactivate ValoFly Tether.Solutions ground station
*   - close connection to ValoFly Tether.Solutions ground station
*/
int main(int argc, char* argv[])
{    
	// new instance of TetherCom object which handle the connection to the tether ground station
	VALOFLY::TetherCom tetherCom;

	// catch help flag
	try {
		parseArguments(argc, argv);
	}
	catch (const std::exception & ex)
	{
		std::cout << "Failed to init TetherCom application!" << std::endl;
		std::cout << ex.what() << std::endl;
		return EXIT_FAILURE;
	}

	printHeader();

	// get a list (vector of string) of available serial ports, with its description and hardware ID
	std::vector<std::string> serialDevListPort;
	std::vector<std::string> serialDevListDesc;
	std::vector<std::string> serialDevListHwID;
	tetherCom.getSerialPortsInfo(serialDevListPort, serialDevListDesc, serialDevListHwID);

	// print all available serial ports, with its description and hardware ID to current standard output
    std::cout << "Available serial ports:" << std::endl;
	tetherCom.printSerialPortList();

    
	// define serial port to connect with by user input
	std::string portName;
	std::cout << std::endl << "Type COM port name to connect." << std::endl;
	std::cin >> portName;

	std::cout << std::endl << "Try to connect to " << portName << "." << std::endl;

	// try to establish serial connection to tether ground station
	try
	{
		tetherCom.openConnection(portName);
	}
	catch (std::exception ex)
	{
		printf("%s", ex.what());
		return EXIT_FAILURE;
	}
	std::cout << "Open serial port \"" << portName << "\" successfull!" << std::endl;

	// configure individual connection timeout in milli seconds to prevent communication issues
	tetherCom.setConnectionTimeout(300);
	
	VALOFLY::TetherCom::systemStatus status;
	try {
		// get current state of tether ground station
		std::cout << "Request tether system status..." << std::endl;
		tetherCom.getSystemStatus(status);


		// print current state of tether ground station to given output stream
		tetherCom.print(status, std::cout);


		// set config of tether ground station
		VALOFLY::TetherCom::systemConfig config;
		config.cableRetractionStartLength = 5.678f; // m
		config.cableRetractionTorque = 50.4f; // %
		config.systemActivation = false; // true/false => on/off
		tetherCom.setSystemConfig(config);


		// set single config parameters - start length of cable retraction
		config.cableRetractionStartLength = 4.123f; // m
		tetherCom.setCableRetractionStartLength(config.cableRetractionStartLength);


		// set single config parameters - torque of cable retraction in percent
		// it is not neccessary to create/set/update a VALOFLY::TetherCom::systemConfig object
		// values can setted directly
		tetherCom.setCableRetractionTorque(40.3f); // %


		// set single config parameters - set system activation
		if (config.systemActivation)
			std::cout << "Activate tether system..." << std::endl;
		else
			std::cout << "Deactivate tether system..." << std::endl;

		tetherCom.setSystemActivation(config.systemActivation);


		// enable tether ground station
		std::cout << "Activate tether system..." << std::endl;
		tetherCom.setSystemActivation(true);


		// disable tether ground station
		std::cout << "Deactivate tether system..." << std::endl;
		tetherCom.setSystemActivation(false);


		// close connection to tether ground station and disconnect from serial port
		tetherCom.closeConnection();
	}
	catch (VALOFLY::IOException ex)
	{
		printf("Error - IOException occurred!\n");
		printf("%s\n", ex.what());
		std::cout << "Close connection..." << std::endl;
		tetherCom.closeConnection();
		return EXIT_FAILURE;
	}
	catch (VALOFLY::DataException ex)
	{
		printf("Error - DataException occurred!\n");
		printf("%s\n", ex.what());
		std::cout << "Close connection..." << std::endl;
		tetherCom.closeConnection();
		return EXIT_FAILURE;
	}
	catch (std::exception ex)
	{
		printf("Error - general exception occurred!\n");
		printf("%s\n", ex.what());
		std::cout << "Close connection..." << std::endl;
		tetherCom.closeConnection();
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}