//
// Copyright (C) ValoFly GmbH - All Rights Reserved
//

#ifndef _VALOFLY_TETHERCOM_STATES_HEADER_
#define _VALOFLY_TETHERCOM_STATES_HEADER_

#include <stdlib.h>
#include <stdint.h>

/******************************************************************/
/*                Public system state definitions                 */
/******************************************************************/

namespace VALOFLY {
    namespace TetherComStates {

        #pragma region Lib_Communication
        /**
        * Enum to classify connection status to ValoFly Tether\.Solutions ground station.
        */
        enum class connectionStatus : uint8_t
        {
            disconnected = 0x00,    /**< No connection established */
            connected = 0x01,       /**< Connection established */
            error = 0xFF            /**< Error during last connection */
        };

        /**
        * Enum to classify states of system status group objects
        */
        enum class statusConfig : uint8_t
        {
            disable = 0x00,         /**< status object is disabled by configuration */
            enable = 0x01,          /**< status object is enabled by configuration */
            not_available = 0xFF    /**< status object is not available on this ValoFly Tether\.Solutions ground station */
        };

        /**
        * Enum to classify communication status to and from ValoFly Tether\.Solutions ground station.
        */
        enum class communicationStatus : uint8_t
        {
            ack = 0x00,
            buffer_overflow = 0x01,
            timeout = 0x02,
            data_format_error = 0x03,
            command_error = 0x04,
            data_error = 0x05,
            command_exec_error = 0x06,
            not_defined = 0xFF
        };
        #pragma endregion

        #pragma region System
        /**
        * Enum to classify error states of ValoFly Tether\.Solutions ground station.
        */
        enum class systemError : uint8_t
        {
            no_error = 0x00,      /**< ValoFly Tether.Solutions system is ok and ready to use */
            generic_error = 0x01  /**< ValoFly Tether.Solutions system generic failure occurred */
        };

        /**
        * Enum to classify activation status of ValoFly Tether\.Solutions ground station.
        */
        enum class systemActivationStatus : uint8_t
        {
            off = 0x00,         /**< Ground station is deactivated [off] */
            manual = 0x01,      /**< Ground station is manual activated by power button [on] */
            remote_uart = 0x10  /**< Ground station is remotely activated via UART communication [on] */
        };

        /**
        * Enum to classify equiped network type ValoFly Tether\.Solutions ground station.
        */
        enum class systemNetworkType : uint8_t
        {
            p2w = 0x01,
            p3w = 0x02,
            fo = 0x03,
            not_available = 0xFF
        };

        /**
        * Enum to classify network connection status of ValoFly Tether\.Solutions ground station.
        */
        enum class networkStatus : uint8_t
        {
            offline = 0x00,
            error = 0x01,
            pairing = 0x02,
            connected = 0x02,
            not_available = 0xFF
        };

        /**
        * Enum to classify memory status of ValoFly Tether\.Solutions ground station.
        */
        enum class memoryStatus : uint8_t
        {
            ok = 0x00,
            error = 0x01,
            overflow = 0x02,
            not_available = 0xFF
        };
        #pragma endregion

        #pragma region Cable
        /**
        * Enum to classify tether cable status of ValoFly Tether\.Solutions ground station.
        */
        enum class cableTetherStatus : uint8_t
        {
            ok = 0x00,              /**< Tether Cable is okay and ready to use */
            maxLength = 0x01,       /**< maximum length is unwounded */
            failure = 0x02,         /**< Tether Cable has a material failure */
        };

        /**
        * Enum to classify cable guide status of ValoFly Tether\.Solutions ground station.
        */
        enum class cableGuideStatus : uint8_t
        {
            ok = 0x00,              /**< Cable guide is okay and ready to use */
            motor_failure = 0x01,   /**< Cable guide motor failure occurred */
            not_available = 0xFF,   /**< Cable guide is not available on this ValoFly Tether\.Solutions ground station */
        };

        /**
        * Enum to classify cable reel status of ValoFly Tether\.Solutions ground station.
        */
        enum class cableReelStatus : uint8_t
        {
            ok = 0x00,              /**< Cable reel is okay and ready to use */
            motor_failure = 0x01,   /**< Cable reel motor failure occurred */
            not_available = 0xFF,   /**< Cable reel is not available on this ValoFly Tether\.Solutions ground station */
        };

        /**
        * Enum to classify cable retraction status of ValoFly Tether\.Solutions ground station.
        */
        enum class cableRetractionStatus : uint8_t
        {
            off = 0x00,             /**< Cable retraction is deactivated [off] */
            manual = 0x01,          /**< Cable retraction is manual activated [on] */
            automatic = 0x02,       /**< Cable retraction is in automatic mode controlled on ground station front panel [on] */
            remote_uart = 0x10,     /**< Cable retraction is remotely controlled via UART communication [on/off] */
            not_available = 0xFF    /**< Cable retraction is not available on this ValoFly Tether\.Solutions ground station */
        };
        #pragma endregion

        #pragma region Environment
        /**
        * Enum to classify environmental sensor status of ValoFly Tether\.Solutions ground station.
        */
        enum class envSensorStatus : uint8_t
        {
            ok = 0x00,              /**< temperature sensor is okay and ready to use */
            sensor_failure = 0x01,  /**< temperature sensor general sensor failure occurred */
            temp_failure = 0x02,    /**< temperature sensor failure occurred */
            humidity_failure = 0x03,/**< humidity sensor failure occurred */
            pressure_failure = 0x04 /**< pressure sensor failure occurred */
        };

        /**
        * Enum to classify fan status of ValoFly Tether\.Solutions ground station.
        */
        enum class fanStatus : uint8_t
        {
            ok = 0x00,              /**< fan is okay and ready to use */
            failure = 0x01,         /**< fan failure occurred */
            not_available = 0xFF    /**< fan is not available on this ValoFly Tether\.Solutions ground station */
        };
        #pragma endregion

        #pragma region Energy
        /**
        * Enum to classify power supply status of ValoFly Tether\.Solutions ground station.
        */
        enum class supplyStatus : uint8_t
        {
            ok = 0x00,
            general_error = 0x01,
            range_error = 0x02,
            not_available = 0xFF
        };

        /**
        * Enum to classify external UAV power supply status of ValoFly Tether\.Solutions ground station.
        */
        enum class extUAVSupplyStatus : uint8_t
        {
            idle = 0x00,
            pri_starting = 0x01,
            pri_running = 0x02,
            stopping = 0x04,
            supply_error = 0x08,
            load_error = 0x10,
            sec_starting = 0x21,
            sec_running = 0x22
        };

        /**
        * Enum to classify current backup battery status of ValoFly Tether\.Solutions ground station.
        */
        enum class backupBatteryStatus : uint8_t
        {
            idle = 0x00,            /**< Backup battery is fine and currently not in use */
            charge = 0x01,          /**< Backup battery is fine and currently in charging progress */
            discharge = 0x02,       /**< Backup battery is fine and currently used to power ValoFly Tether\.Solutions ground station base functionality */
            failure = 0x7F,         /**< Backup battery has failure and is not longer useable */
            not_available = 0xFF    /**< No backup battery available in this ValoFly Tether\.Solutions ground station configuration */
        };
        #pragma endregion
    };
};

#endif // _VALOFLY_TETHERCOM_STATES_HEADER_