//
// Copyright (C) ValoFly GmbH - All Rights Reserved
//
/** \file
* Definition of system state values of ValoFly Tether\.Solutions Ground Station
*
* TetherComDefinitions defines values for system states of ValoFly Tether\.Solutions Ground Station. For a simple interpretation of numerical communicated values enumerates are used to described system states.
*/

#ifndef _VALOFLY_TETHERCOM_DEFINITIONS_CXX_HEADER_
#define _VALOFLY_TETHERCOM_DEFINITIONS_CXX_HEADER_

#include <stdlib.h>
#include <stdint.h>

#include "TetherComDefinitionsVersion.h"
/**********************************************************************/
/*                Public TetherCom system definitions                 */
/**********************************************************************/

/// Namespace for classes and members developed by ValoFly GmbH
namespace VALOFLY {
    /// Namespace for constants to represent system states of ValoFly Tether\.Solutions Ground Station which are used in communication based on VALOFLY::TetherCom Library
    namespace TetherComDefinitions {

        /**
        * Enum to classify connection status to ValoFly Tether\.Solutions Ground Station.
        */
        enum class connectionStatus : uint8_t
        {
            disconnected = 0x00, /**< No connection established */
            connected = 0x01, /**< Connection established */
            error = 0xFF, /**< Error during last connection */
        };

        /**
        * Enum to classify states of system status group objects
        */
        enum class statusConfig : uint8_t
        {
            disable = 0x00, /**< status object is disabled by configuration */
            enable = 0x01, /**< status object is enabled by configuration */
            not_available = 0xFF, /**< status object is not available on this ValoFly Tether\.Solutions Ground Station */
        };

        /**
        * Enum to classify communication status to and from ValoFly Tether\.Solutions Ground Station.
        */
        enum class communicationStatus : uint8_t
        {
            ack = 0x00, /**< Received message is acknowledged */
            buffer_overflow = 0x01, /**< Buffer overflow occured during receiving last message */
            timeout = 0x02, /**< Timeout occured during receiving last message */
            data_format_error = 0x03, /**< Corrupted data format of last received message */
            command_error = 0x04, /**< Unknown command in last received message */
            data_error = 0x05, /**< Corrupted data in last received message (e.g. data does not fit to given command) */
            command_exec_error = 0x06, /**< Error during execution of command of last received message */
            not_defined = 0xFF, /**< Communication with ValoFly Tether\.Solutions Ground Station reach an undefined state */
        };

        /**
        * Enum to classify error states of ValoFly Tether\.Solutions Ground Station.
        */
        enum class systemError : uint8_t
        {
            no_error = 0x00, /**< ValoFly Tether.Solutions system is ok and ready to use */
            generic_error = 0x01, /**< ValoFly Tether.Solutions system generic failure occurred */
        };

        /**
        * Enum to classify activation status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class systemActivationStatus : uint8_t
        {
            off = 0x00, /**< Ground station is deactivated [off] */
            manual = 0x01, /**< Ground station is manual activated by power button [on] */
            remote_uart = 0x10, /**< Ground station is remotely activated via UART communication [on] */
        };

        /**
        * Enum to classify equiped network type ValoFly Tether\.Solutions Ground Station.
        */
        enum class systemNetworkType : uint8_t
        {
            p2w = 0x01, /**< 2-wire Power Line */
            p3w = 0x02, /**< 3-wire Power Line */
            fo = 0x03, /**< Fiber-optic */
            not_available = 0xFF, /**< No network on this ValoFly Tether\.Solutions System available */
        };

        /**
        * Enum to classify network connection status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class networkStatus : uint8_t
        {
            offline = 0x00, /**< Network is offline */
            error = 0x01, /**< Network is in error state */
            pairing = 0x02, /**< Network components try to pair */
            connected = 0x03, /**< Network is connected */
            not_available = 0xFF, /**< No network on this ValoFly Tether\.Solutions System available */
        };

        /**
        * Enum to classify memory status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class memoryStatus : uint8_t
        {
            ok = 0x00, /**< Memory is full usable */
            error = 0x01, /**< Error on memory block */
            overflow = 0x02, /**< Memory overflow */
            not_available = 0xFF, /**< No Memory module on this ValoFly Tether\.Solutions System available */
        };

        /**
        * Enum to classify tether cable status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class cableTetherStatus : uint8_t
        {
            ok = 0x00, /**< Tether Cable is okay and ready to use */
            warning_length = 0x01, /**< 90% length is unwounded */
            max_length = 0x02, /**< maximum length is unwounded */
            failure = 0x03, /**< Tether Cable has a material failure */
        };

        /**
        * Enum to classify cable guide status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class cableGuideStatus : uint8_t
        {
            ok = 0x00, /**< Cable guide is okay and ready to use */
            motor_failure = 0x01, /**< Cable guide motor failure occurred */
            not_available = 0xFF, /**< Cable guide is not available on this ValoFly Tether\.Solutions Ground Station */
        };

        /**
        * Enum to classify cable reel status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class cableReelStatus : uint8_t
        {
            ok = 0x00, /**< Cable reel is okay and ready to use */
            motor_failure = 0x01, /**< Cable reel motor failure occurred */
            not_available = 0xFF, /**< Cable reel is not available on this ValoFly Tether\.Solutions Ground Station */
        };

        /**
        * Enum to classify cable retraction status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class cableRetractionStatus : uint8_t
        {
            off = 0x00, /**< Cable retraction is deactivated [off] */
            manual = 0x01, /**< Cable retraction is manual activated [on] */
            automatic = 0x02, /**< Cable retraction is in automatic mode controlled on Ground Station front panel [on] */
            remote_uart = 0x10, /**< Cable retraction is remotely controlled via UART communication [on/off] */
            not_available = 0xFF, /**< Cable retraction is not available on this ValoFly Tether\.Solutions Ground Station */
        };

        /**
        * Enum to classify environmental sensor status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class envSensorStatus : uint8_t
        {
            ok = 0x00, /**< temperature sensor is okay and ready to use */
            sensor_failure = 0x01, /**< temperature sensor general sensor failure occurred */
            temp_failure = 0x02, /**< temperature sensor failure occurred */
            humidity_failure = 0x03, /**< humidity sensor failure occurred */
            pressure_failure = 0x04, /**< pressure sensor failure occurred */
        };

        /**
        * Enum to classify fan status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class fanStatus : uint8_t
        {
            ok = 0x00, /**< fan is okay and ready to use */
            failure = 0x01, /**< fan failure occurred */
            not_available = 0xFF, /**< fan is not available on this ValoFly Tether\.Solutions Ground Station */
        };

        /**
        * Enum to classify power supply status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class supplyStatus : uint8_t
        {
            ok = 0x00, /**< Supply is usable and in defined range */
            general_error = 0x01, /**< Supply is in error state */
            range_error = 0x02, /**< Supply operate out of range */
            not_available = 0xFF, /**< Supply is not available on this ValoFly Tether\.Solutions Ground Station */
        };

        /**
        * Enum to classify external UAV power supply status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class extUAVSupplyStatus : uint8_t
        {
            inactive = 0x00, /**< External UAV Supply is disabled and ready for activation */
            pri_starting = 0x01, /**< Primary External UAV Supply is starting */
            pri_running = 0x02, /**< Primary External UAV Supply is active */
            stopping = 0x04, /**< External UAV Supply is stopping */
            supply_error = 0x08, /**< Error on External UAV Supply */
            load_error = 0x10, /**< No load at External UAV Supply detected */
            sec_starting = 0x21, /**< Secondary External UAV Supply is starting */
            sec_running = 0x22, /**< Secondary External UAV Supply is stopping */
        };

        /**
        * Enum to classify current backup battery status of ValoFly Tether\.Solutions Ground Station.
        */
        enum class backupBatteryStatus : uint8_t
        {
            idle = 0x00, /**< Backup battery is fine and currently not in use */
            charge = 0x01, /**< Backup battery is fine and currently in charging progress */
            discharge = 0x02, /**< Backup battery is fine and currently used to power ValoFly Tether\.Solutions Ground Station base functionality */
            failure = 0x7F, /**< Backup battery has failure and is not longer useable */
            not_available = 0xFF, /**< No backup battery available in this ValoFly Tether\.Solutions Ground Station configuration */
        };
    };
};

#endif // _VALOFLY_TETHERCOM_DEFINITIONS_CXX_HEADER_